//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.0

import wesual.data.Core 1.0
import wesual.Controls 1.0


PopupAnchorItem {
    id : popup

    property list<Language>  listing
    property int  activeIndex : -1
    property bool closedOnTab : false

    anchors.centerIn : parent

    signal accepted(int result)

    enabled : visible

    popup : Item {
        id : content

        width  : itemColumn.width  + 2 * bg.border.width
        height : itemColumn.height + 2 * bg.border.width

        anchors {
            top : popup.popupVisible ? parent.bottom : undefined
            right : popup.popupVisible ? parent.right : undefined
            rightMargin : -popup.parent.width / 2
            topMargin : popup.parent.height / 2
        }

        opacity : 0

        states : [
            State {
                name : "visible"
                when : popup.popupVisible

                PropertyChanges {
                    target : content
                    opacity : 1
                }
            }
        ]

        transitions : Transition {
            NumberAnimation {
                property : "opacity"
                duration : 200
            }
        }

        // key handling
        Keys.onEscapePressed : {
            popup.hide();
        }
        Keys.onReturnPressed : {
            popup.accepted(popup.activeIndex >= 0 ?
                           popup.activeIndex : undefined)
        }
        Keys.onEnterPressed : {
            popup.accepted(popup.activeIndex >= 0 ?
                           popup.activeIndex : undefined)
        }
        Keys.onUpPressed : {
            if (listing.length < 1)
                return;

            var index = popup.activeIndex - 1;
            if (index < 0)
                index = listing.length - 1;
            popup.activeIndex = index;
        }
        Keys.onDownPressed : {
            if (listing.length < 1)
                return;

            popup.activeIndex = (popup.activeIndex + 1) % listing.length;
        }
        Keys.onRightPressed : {}
        Keys.onLeftPressed : {}
        Keys.onTabPressed  : {
            closedOnTab = true;
            popup.hide();

            var next = nextItemInFocusChain(true);
            if (next)
                next.forceActiveFocus();
        }
        Keys.onBacktabPressed : {
            closedOnTab = true;
            popup.hide();
            var next = nextItemInFocusChain(false);
            if (next)
                next.forceActiveFocus();
        }
        Keys.onPressed : {
            if (event.key === Qt.Key_PageUp || event.key === Qt.Key_PageDown) {
                // do not propagate navigation keys while focus item
                event.accepted = true;
            }
        }
        Rectangle {
            id : bg

            anchors.fill : parent
            border {
                color : UI.color(UI.PrimaryControlBorder)
                width : 1
            }
        }

        Column {
            id : itemColumn

            readonly property int margin : 8

            readonly property int idealWidth : {
                var max = popup.parent ?
                            (popup.parent.width - bg.border.width * 2) : 0;
                for (var i = 0; i < children.length; ++i) {
                    // popup width should at least be 150
                    max = Math.max(max, children[i].implicitWidth, 150);
                }
                return max;
            }

            x : bg.border.width
            y : bg.border.width
            width : idealWidth
            height : childrenRect.height

            Item {
                width : itemColumn.idealWidth
                height : 30

                Label {
                    text : qsTrId("4b2bfa6d924a6566")
                    anchors {
                        verticalCenter : parent.verticalCenter
                        left : parent.left
                        leftMargin : 10
                    }
                }
            }

            Repeater {
                model : popup.listing
                delegate : Item {

                    readonly property int itemIndex : index
                    readonly property bool active : activeIndex === index

                    implicitWidth : label.implicitWidth + 2 * itemColumn.margin
                    width : itemColumn.idealWidth
                    height : 26

                    Rectangle {
                        id : itemBackground

                        height : parent.height
                        width : itemColumn.idealWidth
                        color : UI.color(UI.PrimaryBase)

                        Behavior on color {
                            ColorAnimation {
                                duration : 0
                            }
                        }
                        states : [
                            State {
                                name : "selected"
                                when : active && mouseArea.pressed

                                PropertyChanges {
                                    target : itemBackground
                                    color : UI.color(UI.PrimaryPress)
                                }
                            },
                            State {
                                name : "hovered"
                                when : active

                                PropertyChanges {
                                    target : itemBackground
                                    color  : UI.color(UI.PrimaryHover)
                                }
                            }
                        ]
                    }

                    Item {
                        id : label

                        implicitWidth : flag.width + languageName.implicitWidth
                        width : itemColumn.idealWidth
                        height : 26

                        Image {
                            id : flag
                            width : 22
                            height : 16
                            anchors {
                                verticalCenter : parent.verticalCenter
                                left : parent.left
                                leftMargin : 5
                            }
                            source : modelData.flag
                            sourceSize {
                                width : 24
                                height : 24
                            }
                            fillMode : Image.PreserveAspectCrop
                        }

                        Text {
                            id : languageName
                            text : modelData.name
                            width : parent.width - 50
                            font {
                                family : UI.fontFamily(UI.PrimaryFont)
                                weight : UI.fontWeight(UI.PrimaryFont)
                                pixelSize : 14
                            }
                            anchors {
                                left : flag.right
                                leftMargin : 6
                                verticalCenter : parent.verticalCenter
                            }
                            color : itemBackground.state === "selected" ?
                                    UI.color(UI.SelectedTextColor) :
                                    UI.color(UI.PrimaryTextColor)
                        }
                    }
                }
            }

            // 3px space after last element
            Item {
                width   : itemColumn.idealWidth
                height  : 3
            }
        }

        MouseArea {
            id : mouseArea
            anchors.fill : parent
            acceptedButtons : Qt.LeftButton
            hoverEnabled : true
            onPositionChanged : {
                var child = itemColumn.childAt(mouse.x, mouse.y);
                if (child && child.itemIndex !== undefined) {
                    popup.activeIndex = child.itemIndex;
                } else {
                    popup.activeIndex = -1;
                }
            }
            onReleased : popup.accepted(popup.activeIndex)
        }
    }
}
